--------------------------------------------------------------------------------
-- DIO2 board LCD driver
-- top level
--
-- Michal TRS
-- trsm1@fel.cvut.cz 
--------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity lcd is
  Port (
   CLK         : in std_logic;
   AS_RESET    : in std_logic; 
   -- ------------------------
   OPER        : in std_logic_vector(1 downto 0);
   -- 00 - clear display
   -- 01 - return cursor home
   -- 10 - set DDRAM address
   -- 11 - write ASCII
   ASCII       : in std_logic_vector(7 downto 0);
   VLD         : in std_logic;
   RDY         : out std_logic;
   -- low level
   DATA        : out std_logic_vector(7 downto 0);
   RS          : out std_logic;
   RW          : out std_logic;
   E           : out std_logic);
   end lcd;

architecture lcd_top_level of lcd is

   component lcd_time_drv
     Generic (
      T     : integer := 20;
      tsu   : integer := 60;
      tw    : integer := 300;
      tc    : integer := 600);
     Port (
      CLK      : in std_logic;
      AS_RESET : in std_logic; 
      -- ---------------------
      DATA_IN     : in std_logic_vector(7 downto 0);
      RW_IN       : in std_logic;
      RS_IN       : in std_logic;
      DATA_VLD    : in std_logic;
      -- ------------------------
      DATA_OUT    : out std_logic_vector(7 downto 0);
      RW_OUT      : out std_logic;
      RS_OUT      : out std_logic;
      E           : out std_logic;
      -- -------------------------   
      RDY         : out std_logic);
   end component;

   component lcd_init_drv
     Port ( 
      CLK      : in std_logic;
      AS_RESET : in std_logic;
      -- ---
      INIT     : in  std_logic;
      RDY      : out std_logic;

      DATA     : out std_logic_vector(7 downto 0);
      RW       : out std_logic;
      RS       : out std_logic;
      DATA_VLD : out std_logic;
      WR_RDY   : in  std_logic);
   end component;

   -- interconnection signals
   signal init_start : std_logic;
   signal init_rdy   : std_logic;

   signal llvl_data  : std_logic_vector(7 downto 0);
   signal llvl_rw    : std_logic;
   signal llvl_rs    : std_logic;
   signal llvl_vld   : std_logic;
   signal llvl_rdy   : std_logic;

   signal init_data  : std_logic_vector(7 downto 0);
   signal init_rw    : std_logic;
   signal init_rs    : std_logic;
   signal init_vld   : std_logic;

   signal as_reset_reg  : std_logic;


begin

   i_time: lcd_time_drv 
   Generic map (
      T     => 20,
      tsu   => 40,
      tw    => 220,
      tc    => 500)
   Port map(
      CLK      => CLK,
      AS_RESET => AS_RESET,
      DATA_IN  => llvl_data,
      RW_IN    => llvl_rw,
      RS_IN    => llvl_rs,
      DATA_VLD => llvl_vld,
      DATA_OUT => DATA,
      RW_OUT   => RW,
      RS_OUT   => RS,
      E        => E,
      RDY      => llvl_rdy);

   i_init: lcd_init_drv
   Port map( 
      CLK      => CLK,
      AS_RESET => AS_RESET,
      INIT     => init_start,
      RDY      => init_rdy,
      DATA     => init_data,
      RW       => init_rw,
      RS       => init_rs,
      DATA_VLD => init_vld,
      WR_RDY   => llvl_rdy);


   RDY <= init_rdy and llvl_rdy;

   ----------------------------------------------
   -- switch to init state
   process(CLK)
   begin
      if CLK = '1' and CLK'event then
         as_reset_reg <= AS_RESET;
      end if;   
   end process;

   init_start <= not AS_RESET and as_reset_reg;

   process(init_rdy, init_data, init_rw, init_rs, init_vld, llvl_rdy, ASCII, VLD, OPER)
   begin
      if init_rdy = '0' then
         llvl_data   <= init_data;
         llvl_rw     <= init_rw;
         llvl_rs     <= init_rs;
         llvl_vld    <= init_vld;
      else
      -- ----------------------------------------
      -- OPER switching         
         llvl_vld    <= VLD;
         llvl_rw     <= '0';
         
         case OPER is
            when "00" =>
               -- clear display
               llvl_data   <= "00000001";
               llvl_rs     <= '0';               
            when "01" =>
               -- return cursor home
               llvl_data   <= "00000010";
               llvl_rs     <= '0';
            when "10" =>   
               -- set DDRAM address
               llvl_data   <= '1' & ASCII(6 downto 0);
               llvl_rs     <= '0';
            when "11" =>
               -- write ASCII
               llvl_data   <= ASCII;
               llvl_rs     <= '1';          
            when others =>
               null;
         end case;         
      end if;
   end process;

end lcd_top_level;
