-- cpu.vhd - PicoBlaze with input / output logic for GPS reciever
-- 36APS project
-- Milos Hrdy (hrdym1@fel.cvut.cz), Michal Trs (trsm1@fel.cvut.cz)
-- 

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity CPU is
	Port(
		CLK 	: in std_logic;
		RESET	: in std_logic;

		-- CPU PORT_ID
		-- port		type
		-------------------------
		-- 0x01		STATUS
		-- 0x02		SRAM
		-- 0x04		COM
		-- 0x08		GPS
		-- 0x10		GLCD
		-------------------------

		-- SRAM
		-- TX
		SRAM_TX_DATA	: out std_logic_vector(7 downto 0);
		SRAM_TX_VALID	: out std_logic;
		SRAM_TX_ACK		: in	std_logic;
		-- RX
		SRAM_RX_DATA	: in  std_logic_vector(7 downto 0);
		SRAM_RX_VALID	: in  std_logic;
		SRAM_RX_ACK		: out	std_logic;

		-- COM
		-- TX
		COM_TX_DATA		: out std_logic_vector(7 downto 0);
		COM_TX_VALID	: out std_logic;
		COM_TX_ACK		: in	std_logic;
		-- RX
		COM_RX_DATA		: in  std_logic_vector(7 downto 0);
		COM_RX_VALID	: in  std_logic;
		COM_RX_ACK		: out	std_logic;

		-- GPS
		-- TX
		GPS_TX_DATA		: out std_logic_vector(7 downto 0);
		GPS_TX_VALID	: out std_logic;
		GPS_TX_ACK		: in	std_logic;
		-- RX
		GPS_RX_DATA		: in  std_logic_vector(7 downto 0);
		GPS_RX_VALID	: in  std_logic;
		GPS_RX_ACK		: out	std_logic;

		-- GLCD
		GLCD_TX_DATA	: out std_logic_vector(7 downto 0);
		GLCD_TX_VALID	: out std_logic;
		GLCD_TX_ACK		: in	std_logic
	);

end CPU;


architecture CPU_BODY of CPU is

   component kcpsm3
    Port (
		address 			: out std_logic_vector(9 downto 0);
		instruction 	: in  std_logic_vector(17 downto 0);
		port_id 			: out std_logic_vector(7 downto 0);
		write_strobe 	: out std_logic;
		out_port 		: out std_logic_vector(7 downto 0);
		read_strobe 	: out std_logic;
		in_port 			: in  std_logic_vector(7 downto 0);
		interrupt 		: in  std_logic;
		interrupt_ack 	: out std_logic;
		reset 			: in  std_logic;
		clk 				: in  std_logic
	 );
   end component;

	component app_rom
	 Port (
		address 			: in  std_logic_vector(9 downto 0);
		instruction 	: out std_logic_vector(17 downto 0);
		clk 				: in  std_logic	
	  );
	end component;

	component sped_rom
	 Port (
		address 			: in  std_logic_vector(9 downto 0);
		instruction 	: out std_logic_vector(17 downto 0);
		clk 				: in  std_logic	
	  );
	end component;



	signal pb_address 		: std_logic_vector(9 downto 0);
   signal pb_instruction 	: std_logic_vector(17 downto 0);
   signal pb_port_id 		: std_logic_vector(7 downto 0);
   signal pb_write_strobe 	: std_logic;
   signal pb_out_port 		: std_logic_vector(7 downto 0);
   signal pb_read_strobe 	: std_logic;
   signal pb_in_port 		: std_logic_vector(7 downto 0);  

	signal status				: std_logic_vector(7 downto 0);	-- cteni TX_ACK a RX_VALID 
	signal vld_reg				: std_logic_vector(3 downto 0);
	signal ack_reg				: std_logic_vector(2 downto 0);

	signal banks 				: std_logic_vector(35 downto 0);	 -- 1+5 programu * 18b instrukce = 108
	signal bank_sel			: std_logic;

	-- CPU PORT_ID
	-- port		type
	-------------------------
	-- 0x01		STATUS
	-- 0x02		SRAM
	-- 0x04		COM
	-- 0x08		GPS
	-- 0x10		GLCD
	-------------------------
	constant pSTATUS			: std_logic_vector(7 downto 0) := X"01";
	constant pSRAM				: std_logic_vector(7 downto 0) := X"02";
	constant pCOM				: std_logic_vector(7 downto 0) := X"04";
	constant pGPS				: std_logic_vector(7 downto 0) := X"08";
	constant pGLCD				: std_logic_vector(7 downto 0) := X"10";


begin


  procesor: kcpsm3
    Port map(      
		address			=> pb_address,
		instruction 	=> pb_instruction,
		port_id 			=> pb_port_id,
		write_strobe 	=> pb_write_strobe,
		out_port 		=> pb_out_port,
		read_strobe 	=> pb_read_strobe,
		in_port 			=> pb_in_port,
		interrupt 		=> '0',
		interrupt_ack 	=> open,
		reset 			=> RESET,
		clk 				=> CLK
	);



	app_mem: app_rom
	 Port map (
		address 			=> pb_address,
		instruction 	=> banks(17 downto 0),
		clk 				=> CLK
	  );

	sped_mem: sped_rom
	 Port map (
		address 			=> pb_address,
		instruction 	=> banks(35 downto 18),
		clk 				=> CLK
	  );


	-- --------------------------------------------------
	-- Instruction memory MUX
	-- --------------------------------------------------

	-- MUX sel register
	process(CLK,RESET)
	begin
		if RESET = '1' then
			bank_sel <= '0';
		elsif CLK = '1' and CLK'event then						
			if (pb_address(9 downto 4) = X"3F") then -- 3F0 az 3FF
				bank_sel <= pb_address(0);
			end if;			
		end if;
	end process;


	process(bank_sel,banks)
	begin
  		case bank_sel is
			when '0' =>  
         	pb_instruction <= banks(17 downto 0);   
			when '1' =>  
         	pb_instruction <= banks(35 downto 18);
			when others => 
         	pb_instruction <= banks(17 downto 0);   
		end case;      
	end process;


	-- --------------------------------------------------
	-- ADDRESS DECODER
	-- --------------------------------------------------

	--	STATUS REG
	process(CLK,RESET)
	begin
		if RESET = '1' then
			status <= (others => '0');
		elsif CLK = '1' and CLK'event then
			status(0) <= SRAM_RX_VALID and not ack_reg(0); -- data jsou platna pouze kdyz je vld = 1 a ack = 0
			status(1) <= COM_RX_VALID and not ack_reg(1);  -- pokud ack = 1, data jsou jiz nactena, nutno cist status
			status(2) <= GPS_RX_VALID and not ack_reg(2);
			status(3) <= '0';
			status(4) <= SRAM_TX_ACK nor vld_reg(0);	 -- musi byt vld = 0 a ack = '0' aby se dala zapsat dalsi data
			status(5) <= COM_TX_ACK nor vld_reg(1);   -- nutno v programu cist status pred zapisem
			status(6) <= GPS_TX_ACK nor vld_reg(2);
			status(7) <= GLCD_TX_ACK nor vld_reg(3);
		end if;
	end process;
			  

	-- tx valid reg
	process(CLK,RESET)
	begin
		if RESET = '1' then
			vld_reg <= (others => '0');
		elsif CLK = '1' and CLK'event then
			if pb_write_strobe = '1' then
				case pb_port_id is
					when pSRAM =>
						vld_reg(0) 	<= '1';
						SRAM_TX_DATA <= pb_out_port;
					when pCOM =>
						vld_reg(1) 	<= '1';
						COM_TX_DATA <= pb_out_port;
					when pGPS =>
						vld_reg(2) 	<= '1';
						GPS_TX_DATA <= pb_out_port;
					when pGLCD =>
						vld_reg(3) 	<= '1';
						GLCD_TX_DATA <= pb_out_port;	
					when others => null;				
				end case;
			end if;
				if SRAM_TX_ACK = '1' then
					vld_reg(0) 	<= '0';
				end if;
				if COM_TX_ACK = '1' then
					vld_reg(1) 	<= '0';
				end if;
				if GPS_TX_ACK = '1' then					
					vld_reg(2) 	<= '0';
				end if;
				if GLCD_TX_ACK = '1' then
					vld_reg(3) 	<= '0';					
				end if;
		end if;
	end process;

	SRAM_TX_VALID 	<= vld_reg(0);
	COM_TX_VALID 	<= vld_reg(1);
	GPS_TX_VALID 	<= vld_reg(2);
	GLCD_TX_VALID 	<= vld_reg(3);

	
	-- input mux
	process(pb_port_id,status,SRAM_RX_DATA,COM_RX_DATA,GPS_RX_DATA)
	begin
	   case pb_port_id is
	      when pSTATUS =>
	         pb_in_port <= status;
	      when pSRAM =>
	         pb_in_port <= SRAM_RX_DATA;
	      when pCOM =>
				pb_in_port <= COM_RX_DATA;
	      when pGPS =>
				pb_in_port <= GPS_RX_DATA;
	      when others => 
				pb_in_port <= (others => '0');
	   end case;
	end process;

	-- rx ack logic
	process(CLK,RESET)
	begin
		if RESET = '1' then
			ack_reg <= (others => '0');
		elsif CLK = '1' and CLK'event then
			if pb_read_strobe = '1' then
				case pb_port_id is			      
			      when pSRAM =>
			         ack_reg(0) 	<= '1';	
			      when pCOM =>
						ack_reg(1) 	<= '1';
			      when pGPS =>
						ack_reg(2) 	<= '1';
			      when others => null;
			   end case;
			end if;

			if SRAM_RX_VALID = '0' then
				ack_reg(0) 	<= '0';
			end if;
			if COM_RX_VALID = '0' then
				ack_reg(1) 	<= '0';
			end if;
			if GPS_RX_VALID = '0' then					
				ack_reg(2) 	<= '0';
			end if;				
		end if;
	end process;

	SRAM_RX_ACK	<= ack_reg(0);
	COM_RX_ACK	<= ack_reg(1);
	GPS_RX_ACK	<= ack_reg(2);


end CPU_BODY;
