-- ee_data_int.vhd : Data interface between EE and MSH
-- Copyright (C) 2003 CESNET
-- Author(s): Marek Tomas <marekt@liberouter.org>
--
-- This program is free software; you can redistribute it and/or
-- modify it under the terms of the OpenIPCore Hardware General Public
-- License as published by the OpenIPCore Organization; either version
-- 0.20-15092000 of the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but
-- WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- OpenIPCore Hardware General Public License for more details.
--
-- You should have received a copy of the OpenIPCore Hardware Public
-- License along with this program; if not, download it from
-- OpenCores.org (http://www.opencores.org/OIPC/OHGPL.shtml).
--
-- $Id: ee_data_int.vhd,v 1.5 2005/09/29 14:37:32 cejkar Exp $
-- 

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library work;
use work.types.all;

-- -----------------------------------------------------------------------------
--       Entity :  Data interface between EE and MSH
-- -----------------------------------------------------------------------------
entity ee_data_int is
	port (
		-- clock
		CLK			: in std_logic;		-- MSH's clock
		AS_RESET		: in std_logic;		-- asynchronous reset

		-- signals between EE and MSH
		EE_READ_REQ	: in std_logic;		-- read request from edit engine
		EE_ACK		: out std_logic;		-- edit engine read request acknowledge
		EE_DATA		: out std_logic_vector(EE_DATA_MSB downto 0);
			-- edit engine data bus
		EE_WRITE_REQ_RE: in std_logic;	-- edit engine write request rising edge

		-- signals between ee_addr_int and data_access
		EE_INT_REQ	: out std_logic;		-- read request to data access
		EE_ADDR_OFFSET	: out std_logic_vector(DATA_IN_ADDRESS_MSB downto 0);
			-- edit engine address offset
		EE_INT_ACK: in std_logic;
			-- acknowledge from data access to ee data interface
		EE_INT_DATA: in std_logic_vector(15 downto 0)
			-- data cached from edit engine to memory
	);
end ee_data_int;

-- -----------------------------------------------------------------------------
--       Architecture :     Structural level
-- -----------------------------------------------------------------------------
architecture Behavioral of ee_data_int is

	signal ee_addr_offset_i: std_logic_vector(DATA_IN_ADDRESS_MSB downto 0);
		-- address offset in mamory frame

	signal ee_read_req1: std_logic;
	signal ee_read_req2: std_logic;
      -- metastability registers for EE_READ_REQ

   signal ee_int_ack_old: std_logic;
      -- ee_int_ack rising edge detection

begin

	-- --------------------------------------------------------------------------
	-- metastability registers for EE_READ_REQ
	process(CLK,AS_RESET)
	begin
		if AS_RESET='1' then
			ee_read_req1 <= '0';
		elsif CLK='1' and CLK'event then
			ee_read_req1 <= EE_READ_REQ;
		end if;
	end process;

	process(CLK,AS_RESET)
	begin
		if AS_RESET='1' then
			ee_read_req2 <= '0';
		elsif CLK='1' and CLK'event then
			ee_read_req2 <= ee_read_req1;
		end if;
	end process;

	-- --------------------------------------------------------------------------
	-- remember memory offset of processed data
	process(CLK,AS_RESET)
	begin
		if AS_RESET='1' then
			ee_addr_offset_i <= (others=>'0');
		elsif CLK='1' and CLK'event then
			if EE_WRITE_REQ_RE='1' then
				-- new address is registered, offset is set to 0
				ee_addr_offset_i <= (others=>'0');
         elsif ee_int_ack='1' and ee_int_ack_old='0' then
				-- incrementation is enable on ee_int_ack falling edge
				ee_addr_offset_i <= ee_addr_offset_i + 1;
				-- next offset is curent offset incremented by 1
			end if;
		end if;
	end process;
	EE_ADDR_OFFSET <= ee_addr_offset_i;

	-- --------------------------------------------------------------------------
   -- keeps EE_INT_REQ request high until ack is received
   process(CLK,AS_RESET)
   begin
      if AS_RESET='1' then
         EE_INT_REQ <= '0';
      elsif CLK='1' and CLK'event then
--         if ee_int_ack='1' and ee_int_ack_old='0' then
         if ee_int_ack='1' then
            EE_INT_REQ <= '0';
         elsif ee_read_req2='1' then
            EE_INT_REQ <= '1';
         end if;
      end if;
   end process;

	-- --------------------------------------------------------------------------
   -- ee_int_ack rising edge detection
   process(CLK,AS_RESET)
   begin
      if AS_RESET='1' then
         ee_int_ack_old <= '0';
      elsif CLK='1' and CLK'event then
         ee_int_ack_old <= ee_int_ack;
      end if;
   end process;

	-- --------------------------------------------------------------------------
   EE_ACK <= EE_INT_ACK;
   EE_DATA <= EE_INT_DATA;
	-- --------------------------------------------------------------------------

end Behavioral;

