--
-- sdram_ctrl_int.vhd : Interface between SDRAM_CTRL and MSH
-- Copyright (C) 2003-2005  CESNET
-- Author(s): Michal Trs <trsm1@liberouter.org>
--
-- This program is free software; you can redistribute it and/or
-- modify it under the terms of the OpenIPCore Hardware General Public
-- License as published by the OpenIPCore Organization; either version
-- 0.20-15092000 of the License, or (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful, but
-- WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- OpenIPCore Hardware General Public License for more details.
--
-- You should have received a copy of the OpenIPCore Hardware Public
-- License along with this program; if not, download it from
-- OpenCores.org (http://www.opencores.org/OIPC/OHGPL.shtml).
--
-- $Id: sdram_ctrl_int.vhd,v 1.1 2005/12/16 23:30:28 trsm1 Exp $
--

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use IEEE.STD_LOGIC_ARITH.all;

-- pragma translate_off
library UNISIM;
use UNISIM.VComponents.all;
-- pragma translate_on

library work;
use work.types.all;

-- -----------------------------------------------------------------------------
--       Entity declaration: Interface between SDRAM_CTRL and MSH
-- -----------------------------------------------------------------------------

entity sdram_ctrl_int is
   generic (
      SDRAM_ADDR_WIDTH  : integer := 25;
      FAKE_ADDR_WIDTH   : integer := 10
   );
   port (
      -- clock
      CLK               : in std_logic;   -- MSH's clock
      AS_RESET          : in std_logic;   -- asynchronous reset

      -- EE common interface connection
      EE_INT_REQ        : in std_logic;   -- request from EE
      EE_DATA_TR_ACK    : buffer std_logic;  -- EE_INT_REQ acknowledge
      EE_DATA           : out std_logic_vector(EE_INT_DATA_MSB downto 0);
         -- read data burst, data are valid when EE_DATA_TR_ACK is high
      EE_ADDRESS        : in std_logic_vector(PACKET_ADDRESS_MSB downto 0);
         -- address alocated for coresponding EE
      EE_ADDR_OFFSET    : in std_logic_vector(DATA_IN_ADDRESS_MSB downto 0);
         -- offset of read data block

      -- HFE common interface connection
      HFE_INT_REQ       : in std_logic;   -- request from HFE
      -- this signal will be inactivated, write can be done until
      -- fifo is full
      HFE_DATA_TR       : out std_logic;
         -- assembly unit ready to transfer data from HFE interface in
         -- memory output register
      HFE_DATA_TR_ACK   : in std_logic;
         -- acknowledge from HFE on HFE_DATA_TR
      HFE_INT_FIFO_DATA : in std_logic_vector(HFE_INT_DATA_MSB downto 0);
         -- data from HFE
      WFIFO_FULL        : out std_logic;
         -- write fifo is full

      -- DDR common interface connection
      -- Control
      DDR_BUSY          : in std_logic;
      DDR_REQ           : in std_logic;
      DDR_ACK           : out  std_logic;

      -- commands from this unit
      DDR_ADDR_IN       : out  std_logic_vector(SDRAM_ADDR_WIDTH-1 downto 0);
      DDR_WRITE_EN      : out  std_logic;
      DDR_VALID_IN      : out  std_logic;
      DDR_CMD_BUSY      : in std_logic;

      -- data input
      DDR_DATA_IN       : out  std_logic_vector(127 downto 0);
      DDR_DATA_IN_VLD   : out  std_logic;
      DDR_DATA_BUSY     : in std_logic;

      -- data output
      DDR_DATA_OUT      : in std_logic_vector(127 downto 0);
      DDR_DATA_OUT_VLD  : in std_logic;
      DDR_DATA_OUT_NEXT : out  std_logic

      );
end sdram_ctrl_int;

-- -----------------------------------------------------------------------------
--       Architecture :
-- -----------------------------------------------------------------------------
architecture behavioral of sdram_ctrl_int is

   -- controling FSM
   -- signal ee_addr_en       : std_logic; -- out
   signal ee_trans_end     : std_logic; -- in
   signal ee_trans_en      : std_logic; -- out, enable for data transfer

   type t_states is (FSM_idle,FSM_wait,FSM_wait_cmd,FSM_wait_data,FSM_hfe_req,
      FSM_hfe_trans,FSM_ee_req,FSM_ee_trans);

   signal cur_state        : t_states;
   signal next_state       : t_states;

   -- HFE part
   signal hfe_data_tr_ack_re  : std_logic;
   signal hfe_data_tr_ack_fe  : std_logic;
   signal hfe_data_tr_ack_reg : std_logic;

   signal reg_hfe_data     : std_logic_vector(127 downto 0);
   signal reg_hfe_data_ce  : std_logic_vector(3 downto 0);

   signal reg_hfe_addr     : std_logic_vector(SDRAM_ADDR_WIDTH-1 downto 0);

   signal cnt_hfe_out      : std_logic_vector(1 downto 0);

   signal cnt_hfe_en       : std_logic;

   signal reg_hfe_out      : std_logic;

   -- EE part
   signal ee_int_req_re    : std_logic;
   signal ee_int_req_reg   : std_logic;

   signal reg_ee_data      : std_logic_vector(127 downto 0);

   signal ee_addr          : std_logic_vector(SDRAM_ADDR_WIDTH-1 downto 0);
   signal ee_addr_valid    : std_logic;

   signal ddr_load         : std_logic;

   signal cnt_ee_out       : std_logic_vector(2 downto 0);
   signal cnt_ee_out_reg   : std_logic; -- MSB reg of ee_out;
   signal cnt_ee_of        : std_logic; -- counter overflow
   signal cnt_ee_full      : std_logic; -- counter full

   -- second data transfer to ee
   signal ee_sec_run       : std_logic_vector(0 downto 0);

   --common part

   -- address switch
   signal ddr_read         : std_logic;

begin

   -- -------------------------------------------------------------------------
   -- FSM
   -- states:  IDLE, WAIT, WAIT_DATA, WAIT_CMD, HFE_REQ, HFE_TRANS, EE_REQ,
   --    EE_TRANS
   -- inputs:  DDR_REQ, DDR_BUSY, DDR_CMD_BUSY, DDR_DATA_BUSY, HFE_INT_REQ,
   --    HFE_DATA_TR_ACK, EE_INT_REQ, DDR_DATA_OUT_VLD, EE_DATA_TR_ACK
   -- outputs: DDR_ACK, WFIFO_FULL, HFE_DATA_TR, hfe_trans_en, EE_ADDR_EN,
   --    ee_addr_en, EE_DATA_TR_ACK

   -- current state logic
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cur_state <= FSM_idle;
      elsif (CLK'event and CLK = '1') then
         cur_state <= next_state;
      end if;
   end process;

   -- next state logic
   process(cur_state,HFE_INT_REQ,HFE_DATA_TR_ACK,DDR_REQ,DDR_BUSY,EE_INT_REQ,
      DDR_DATA_OUT_VLD,ee_trans_end,DDR_DATA_BUSY,DDR_CMD_BUSY)
   begin
      next_state <= FSM_idle;

      case cur_state is
         when FSM_idle  =>
            if DDR_REQ = '1' then
               next_state <= FSM_wait;
            elsif HFE_INT_REQ = '1' then
               next_state <= FSM_hfe_req;
            elsif EE_INT_REQ = '1' then
               next_state <= FSM_ee_req;
            elsif DDR_CMD_BUSY = '1' then
               next_state <= FSM_wait_cmd;
            elsif DDR_DATA_BUSY = '1' then
               next_state <= FSM_wait_data;
            end if;

         when FSM_wait  =>
            if DDR_BUSY = '1' then
               next_state <= FSM_wait;
            else
               next_state <= FSM_wait;
            end if;

         when FSM_wait_cmd =>
            if DDR_CMD_BUSY = '1' then
               next_state <= FSM_wait_cmd;
            else
               next_state <= FSM_idle;
            end if;

         when FSM_wait_data   =>
            if DDR_DATA_BUSY = '1' then
               next_state <= FSM_wait_data;
            else
               next_state <= FSM_idle;
            end if;

         -- HFE part
         when FSM_hfe_req  =>
            if HFE_DATA_TR_ACK = '0' then
               next_state <= FSM_hfe_req;
            else
               next_state <= FSM_hfe_trans;
            end if;

         when FSM_hfe_trans   =>
            if HFE_DATA_TR_ACK = '1' then
               next_state <= FSM_hfe_trans;
            else
               next_state <= FSM_idle;
            end if;

         -- EE part
         when FSM_ee_req   =>
            if DDR_DATA_OUT_VLD = '0' then
               next_state <= FSM_ee_req;
            else
               next_state <= FSM_ee_trans;
            end if;

         when FSM_ee_trans =>
            if ee_trans_end = '0' then
               next_state <= FSM_ee_trans;
            else
               next_state <= FSM_idle;
            end if;

         when others    =>
            null;

      end case;
   end process;

   -- Ooutput logic
   process (cur_state,HFE_INT_REQ,HFE_DATA_TR_ACK,DDR_REQ,DDR_BUSY,EE_INT_REQ,
      DDR_DATA_OUT_VLD,ee_trans_end,DDR_DATA_BUSY,DDR_CMD_BUSY)
   begin
      -- default settings
      ee_trans_en <= '0';
      HFE_DATA_TR <= '0';
      DDR_ACK     <= '0';
      WFIFO_FULL  <= '0';

      case cur_state is
         when FSM_idle  =>
            DDR_WRITE_EN <= '0';
            if DDR_REQ = '1' then
               DDR_ACK <= '1';
            elsif DDR_CMD_BUSY = '1' then
               WFIFO_FULL <= '1';
            elsif DDR_DATA_BUSY = '1' then
               WFIFO_FULL <= '1';
            elsif HFE_INT_REQ = '1' then
               ddr_read <= '0';
               DDR_WRITE_EN <= '1';
            elsif EE_INT_REQ = '1' then
               ddr_read <= '1';
               DDR_WRITE_EN <= '0';
            end if;

         -- wait until SDRAM_CTRL ready
         when FSM_wait  =>
            WFIFO_FULL <= DDR_BUSY;

         when FSM_wait_cmd =>
            WFIFO_FULL <= DDR_CMD_BUSY;

         when FSM_wait_data   =>
            WFIFO_FULL <= DDR_DATA_BUSY;

         -- HFE part
         when FSM_hfe_req  =>
            HFE_DATA_TR <= '1';

         -- ENABLE for HFE is HFE_DATA_TR_ACK

         -- EE part
         when FSM_ee_req   =>
            WFIFO_FULL <= '1';

         when FSM_ee_trans =>
            ee_trans_en <= '1';
            WFIFO_FULL <= '1';

         when others    =>
            null;

      end case;
   end process;


-- ----------------------------------------------------------------------
-- HFE part


   -- DATA PART
   ------------

   -- input DFF for counter enable
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cnt_hfe_en <= '0';
      elsif CLK = '1' and CLK'event then
         if hfe_data_tr_ack_re = '1' then
            cnt_hfe_en <= '1';
         end if;
         if hfe_data_tr_ack_fe = '1' then
            cnt_hfe_en <= '0';
         end if;
      end if;
   end process;

   -- counter
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cnt_hfe_out <= (others => '0');
      elsif CLK = '1' and CLK'event then
         if cnt_hfe_en = '1' and hfe_data_tr_ack_fe = '0' then
            cnt_hfe_out <= cnt_hfe_out + 1;
         end if;
      end if;
   end process;

   -- decoder
   process(cnt_hfe_out)
   begin
      reg_hfe_data_ce <= (others => '0');
      case cnt_hfe_out is
         when "00"   => reg_hfe_data_ce <= "1000";
         when "01"   => reg_hfe_data_ce <= "0100";
         when "10"   => reg_hfe_data_ce <= "0010";
         when "11"   => reg_hfe_data_ce <= "0001";
         when others => null;
      end case;
   end process;

   -- DFF (set DDR_DATA_IN_VLD)
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         reg_hfe_out <= '0';
      elsif CLK = '1' and CLK'event then
         reg_hfe_out <= reg_hfe_data_ce(0);
      end if;
   end process;

   -- output 4x 32bit DFF
hfe_reg:
   for i in 0 to (128/32)-1 generate
      process(CLK,AS_RESET)
      begin
         if AS_RESET = '1' then
            reg_hfe_data((i+1)*32-1 downto i*32) <= (others => '0');
         elsif CLK = '1' and CLK'event then
            if reg_hfe_data_ce(i) = '1' then
               reg_hfe_data((i+1)*32-1 downto i*32) <= HFE_INT_FIFO_DATA;
            end if;
         end if;
      end process;
   end generate;


   DDR_DATA_IN_VLD   <= reg_hfe_out;
   DDR_DATA_IN       <= reg_hfe_data;


   -- ADDR PART
   ------------

   -- 32b DFF
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         reg_hfe_addr <= (others => '0');
      elsif CLK ='1' and CLK'event then
         if hfe_data_tr_ack_re = '1' then
            reg_hfe_addr <= (others => '0');
            reg_hfe_addr(PACKET_ADDRESS_MSB+DATA_IN_ADDRESS_MSB+3 downto 0) <=
               HFE_INT_FIFO_DATA(PACKET_ADDRESS_MSB+16 downto 16) &
               HFE_INT_FIFO_DATA(DATA_IN_ADDRESS_MSB downto 0) & "00";
         end if;
      end if;
   end process;


   -- common HFE
   -------------

   -- HFE_DATA_TR_ACK RE and FE detector
   process (CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         hfe_data_tr_ack_reg <= '0';
      elsif CLK = '1' and CLK'event then
         hfe_data_tr_ack_reg <= HFE_DATA_TR_ACK;
      end if;
   end process;

   hfe_data_tr_ack_re <= HFE_DATA_TR_ACK and not hfe_data_tr_ack_reg;
   hfe_data_tr_ack_fe <= not HFE_DATA_TR_ACK and hfe_data_tr_ack_reg;

-- ----------------------------------------------------------------------
-- EE part

   -- DATA part

   -- counter 0-7
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cnt_ee_out <= (others => '0');
      elsif CLK = '1' and CLK'event then
         if ee_trans_en = '1' then
            cnt_ee_out <= cnt_ee_out + 1;
         end if;
      end if;
   end process;

   -- counter MSB registred output
   process (CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         cnt_ee_out_reg <= '0';
      elsif CLK = '1' and CLK'event then
         cnt_ee_out_reg <= cnt_ee_out(2);
      end if;
   end process;

   -- counter overflow indicator
   process (cnt_ee_out)
   begin
      if (cnt_ee_out = "000") and (cnt_ee_out_reg = '1') then
        cnt_ee_of <= '1';
      else
        cnt_ee_of <= '0';
      end if;
   end process;

   -- counter full indicator
   process (cnt_ee_out)
   begin
      if cnt_ee_out = "111" then
        cnt_ee_full <= '1';
      else
        cnt_ee_full <= '0';
      end if;
   end process;


   DDR_DATA_OUT_NEXT <= cnt_ee_full;

   --
   process(cnt_ee_out,AS_RESET)
   begin
      if cnt_ee_out = "000" then
         ddr_load <= '1';
      else
         ddr_load <= '0';
      end if;
   end process;

   --ddr_load <= cnt_ee_of;

   -- counter 0-1
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         ee_sec_run <= (others => '0');
      elsif CLK = '1' and CLK'event then
         if cnt_ee_of = '1' and ee_trans_en = '1' then
            ee_sec_run <= ee_sec_run + 1;
         end if;
      end if;
   end process;

   -- detection end of trans
   ee_trans_end <= cnt_ee_full and ee_sec_run(0);


   -- 8x 16b DFF
ee_reg:
   for i in 0 to (128/16)-1 generate
      ee_reg1:
      if i = 0 generate
         process(CLK,AS_RESET)
         begin
            if AS_RESET = '1' then
               reg_ee_data(16*(i+1)-1 downto i*16) <= (others => '0');
            elsif CLK = '1' and CLK'event then
               if ee_trans_en = '1' then
                  reg_ee_data(16*(i+1)-1 downto i*16) <=
                     DDR_DATA_OUT(16*(i+1)-1 downto i*16);
               end if;
            end if;
         end process;
      end generate;

      ee_reg2:
      if i /= 0 generate
         process(CLK,AS_RESET)
         begin
            if AS_RESET = '1' then
               reg_ee_data(16*(i+1)-1 downto i*16) <= (others => '0');
            elsif CLK = '1' and CLK'event then
              if ee_trans_en = '1' then
                if ddr_load = '1' then
                  reg_ee_data(16*(i+1)-1 downto i*16) <=
                     DDR_DATA_OUT(16*(i+1)-1 downto i*16);
                else
                  reg_ee_data(16*(i+1)-1 downto i*16) <=
                     reg_ee_data(16*i-1 downto (i-1)*16);
                end if;
               end if;
            end if;
         end process;
      end generate;
   end generate;

   EE_DATA <= reg_ee_data(127 downto (127+1)-16);

   -- EE_DATA_TR_ACK signal generator;

   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         EE_DATA_TR_ACK <= '0';
      elsif CLK = '1' and CLK'event then
         EE_DATA_TR_ACK <= ee_trans_en;
      end if;
   end process;


   -- ADDR part
   process(EE_ADDRESS,EE_ADDR_OFFSET)
   begin
      --if ee_addr_en ='1' then
         ee_addr <= (others => '0');
         ee_addr(PACKET_ADDRESS_MSB+DATA_IN_ADDRESS_MSB+3 downto 0)
            <= EE_ADDRESS & EE_ADDR_OFFSET & "00" ;
      --end if;
   end process;

   -- EE_INT_REQ RE detector
   process (CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         ee_int_req_reg <= '0';
      elsif CLK = '1' and CLK'event then
         ee_int_req_reg <= EE_INT_REQ;
      end if;
   end process;

   ee_int_req_re <= EE_INT_REQ and not ee_int_req_reg;


   -- valid addr from ee = ee_int_req_re (1 cycle late) -- zeptat se Toma
   process(CLK,AS_RESET)
   begin
      if AS_RESET = '1' then
         ee_addr_valid <= '0';
      elsif CLK = '1' and CLK'event then
         ee_addr_valid <= ee_int_req_re;
      end if;
   end process;

-- ----------------------------------------------------------------------
-- Common part


   -- address switch
   process (ddr_read,reg_hfe_addr,ee_addr)
   begin
      if ddr_read = '1' then
         DDR_ADDR_IN <= ee_addr;       -- EE
      else
         DDR_ADDR_IN <= reg_hfe_addr;  -- HFE
      end if;
   end process;

   -- address valid switch
   process(ddr_read,hfe_data_tr_ack_fe,ee_addr_valid)
   begin
      if ddr_read = '1' then
         DDR_VALID_IN <= ee_addr_valid;
      else
         DDR_VALID_IN <= hfe_data_tr_ack_fe;
      end if;
   end process;


end Behavioral;
